import time
import datetime
import hashlib
from flask_sqlalchemy import SQLAlchemy
from authlib.integrations.sqla_oauth2 import (
    OAuth2ClientMixin,
    OAuth2AuthorizationCodeMixin,
    OAuth2TokenMixin,
)

db = SQLAlchemy()


def salt_pwd(s: str) -> str:
    m = hashlib.md5()
    m.update(s.encode('utf8'))
    return m.hexdigest()


class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(40), unique=True)
    password = db.Column(db.String(255))
    email = db.Column(db.String(255))
    nickname = db.Column(db.String(255))
    usericon = db.Column(db.Text)  # 头像base64编码
    qq_openid = db.Column(db.String(255))  # openid 获取信息
    qq_unionid = db.Column(db.String(255))  # unionid 绑定账号
    cellphone = db.Column(db.String(255))  # +86?
    create_time = db.Column(db.DateTime)

    def __str__(self):
        return self.username

    def get_user_id(self):
        return self.id

    def check_password(self, pwd):
        pwd_check = pwd + str(self.create_time)
        return self.password == salt_pwd(pwd_check)

    def change_password(self, new_pwd):
        pwd_salt = new_pwd + str(self.create_time)
        self.password = salt_pwd(pwd_salt)
        try:
            db.session.commit()
        except Exception:
            db.session.rollback()

    @staticmethod
    def check_user_exist(key='username', value=None):
        check_info = {key: value}
        return User.query.filter_by(**check_info).first()

    @staticmethod
    def create_user(**kwargs):
        time_now = datetime.datetime.now().replace(microsecond=0)
        user_info = {
            'username': kwargs.get('username'),
            'password': salt_pwd(kwargs.get("password") + str(time_now)),
            'email': kwargs.get('email'),
            'cellphone': kwargs.get('cellphone'),
            'qq_openid': kwargs.get('qq_openid'),
            'qq_unionid': kwargs.get('qq_unionid'),
            'create_time': time_now,
        }
        new_user = User(**user_info)
        try:
            db.session.add(new_user)
            db.session.commit()
        except Exception:
            db.session.rollback()
            return None
        return new_user


class OAuth2Client(db.Model, OAuth2ClientMixin):
    __tablename__ = 'oauth2_client'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')


class OAuth2AuthorizationCode(db.Model, OAuth2AuthorizationCodeMixin):
    __tablename__ = 'oauth2_code'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')


class OAuth2Token(db.Model, OAuth2TokenMixin):
    __tablename__ = 'oauth2_token'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')

    def is_refresh_token_active(self):
        if self.revoked:
            return False
        expires_at = self.issued_at + self.expires_in * 2
        return expires_at >= time.time()


class Home(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(255), unique=True)   # uuid4
    nickname = db.Column(db.String(255))
    mac = db.Column(db.String(255))
    owner = db.Column(db.Integer)   # 家庭所有者ID 考虑是否应放在home_user表


user_homes = db.Table("user_home",
                      db.Column("home_id", db.Integer, db.ForeignKey('home.id')),
                      db.Column("user_id", db.Integer, db.ForeignKey('user.id'))
                      )


class Device(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(255))    # 同home下device name 不同？
    mac = db.Column(db.String(255))     # mac 地址
    home_id = db.Column(db.String(255))     # 绑定的家庭id

