import time
import uuid
import datetime
import hashlib
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy.exc import SQLAlchemyError
from authlib.integrations.sqla_oauth2 import (
    OAuth2ClientMixin,
    OAuth2AuthorizationCodeMixin,
    OAuth2TokenMixin,
)
from .generator import *

db = SQLAlchemy()


def salt_pwd(s: str) -> str:
    m = hashlib.md5()
    m.update(s.encode('utf8'))
    return m.hexdigest()


class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(40), unique=True)
    password = db.Column(db.String(255))
    email = db.Column(db.String(255))
    nickname = db.Column(db.String(255))
    usericon = db.Column(db.Text)  # 头像base64编码
    qq_openid = db.Column(db.String(255))  # openid 获取信息
    qq_unionid = db.Column(db.String(255))  # unionid 绑定账号
    cellphone = db.Column(db.String(255))  # +86?
    create_time = db.Column(db.DateTime)

    def __str__(self):
        return self.username

    def get_user_id(self):
        return self.id

    def check_password(self, pwd):
        pwd_check = pwd + str(self.create_time)
        return self.password == salt_pwd(pwd_check)

    def change_password(self, new_pwd):
        pwd_salt = new_pwd + str(self.create_time)
        self.password = salt_pwd(pwd_salt)
        try:
            db.session.commit()
            return True
        except SQLAlchemyError:
            db.session.rollback()
            return False

    @staticmethod
    def check_user_exist(key='username', value=None):
        check_info = {key: value}
        return User.query.filter_by(**check_info).first()

    @staticmethod
    def create_user(**kwargs):
        time_now = datetime.datetime.now().replace(microsecond=0)
        user_info = {
            'username': kwargs.get('username'),
            'password': salt_pwd(kwargs.get("password") + str(time_now)),
            'email': kwargs.get('email'),
            'nickname': kwargs.get('username'),
            'cellphone': kwargs.get('cellphone'),
            'qq_openid': kwargs.get('qq_openid'),
            'qq_unionid': kwargs.get('qq_unionid'),
            'create_time': time_now,
        }
        new_user = User(**user_info)
        try:
            db.session.add(new_user)
            db.session.commit()
            home_name = new_user.username + "的家庭"
            Home.create_home(nickname=home_name, owner_id=new_user.id)
        except SQLAlchemyError:
            db.session.rollback()
            return None
        return new_user

    def check_user_device(self, device_id):
        u_homes = db.session.query(user_homes).filter_by(user_id=self.id).all()
        home_ids = []
        for h in u_homes:
            home_ids.append(h.home_id)
        return Device.query.join(Room).filter(
            Room.home_id.in_(home_ids), Device.id == device_id).first()


class OAuth2Client(db.Model, OAuth2ClientMixin):
    __tablename__ = 'oauth2_client'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')


class OAuth2AuthorizationCode(db.Model, OAuth2AuthorizationCodeMixin):
    __tablename__ = 'oauth2_code'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')


class OAuth2Token(db.Model, OAuth2TokenMixin):
    __tablename__ = 'oauth2_token'

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(
        db.Integer, db.ForeignKey('user.id', ondelete='CASCADE'))
    user = db.relationship('User')

    def is_refresh_token_active(self):
        if self.revoked:
            return False
        expires_at = self.issued_at + self.expires_in * 2
        return expires_at >= time.time()


class Home(db.Model):
    HOME_OWNER = 0
    HOME_ADMIN = 1
    HOME_USER = 10
    __tablename__ = 'home'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(255), unique=True)  # uuid4
    nickname = db.Column(db.String(255), nullable=False)
    longitude = db.Column(db.Float)
    latitude = db.Column(db.Float)
    location_name = db.Column(db.String)
    create_user = db.Column(db.Integer, db.ForeignKey('user.id'))
    create_time = db.Column(db.DateTime, default=datetime.datetime.now)

    @classmethod
    def create_home(cls, owner_id, **kwargs):
        home_id = home_id_generator.generator_id()
        new_home_dict = {
            "id": home_id,
            "name": str(uuid.uuid4()),
            "nickname": kwargs.get("nickname"),
            "create_user": owner_id,
            "longitude": kwargs.get("longitude"),
            "latitude": kwargs.get("latitude"),
            "location_name": kwargs.get("location_name")
        }
        rel_sql = user_homes.insert().values(home_id=home_id, user_id=owner_id, permission=cls.HOME_OWNER)
        new_h = cls(**new_home_dict)
        try:
            db.session.add(new_h)
            db.session.execute(rel_sql)
            db.session.commit()
            # 创建家庭时默认创建房间
            new_h.create_room(room_name="全屋", room_type=Room.DEFAULT_ROOM)
            return new_h
        except Exception as e:
            print(e)
            db.session.rollback()
            return None

    def create_room(self, room_name, room_type=None):
        if room_type is not None:
            return Room.create_room(room_name, self.id, room_type=room_type)
        return Room.create_room(room_name, self.id)


user_homes = db.Table("user_home",
                      db.Column("home_id", db.Integer, db.ForeignKey('home.id')),
                      db.Column("user_id", db.Integer, db.ForeignKey('user.id')),
                      db.Column("permission", db.Integer)
                      )


class Room(db.Model):
    DEFAULT_ROOM = 0
    OTHER_ROOM = 1
    __tablename__ = 'room'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(255), unique=True)  # uuid4
    room_type = db.Column(db.Integer, default=1)
    nickname = db.Column(db.String, nullable=False)
    home_id = db.Column(db.Integer, db.ForeignKey('home.id'), nullable=False)

    @classmethod
    def create_room(cls, room_name, home_id, room_type=1):
        room_id = room_id_generator.generator_id()
        new_room = cls(
            id=room_id, name=str(uuid.uuid4()), nickname=room_name, home_id=home_id, room_type=room_type
        )

        try:
            db.session.add(new_room)
            db.session.commit()
            return new_room
        except Exception as e:
            print(e)
            db.session.rollback()
            return None


class Device(db.Model):
    __tablename__ = 'device'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(255), nullable=False)  # 同home下device name 不同？
    mac = db.Column(db.String(255), nullable=False)  # mac 地址
    is_online = db.Column(db.Boolean, default=False)   # 是否在线
    acc_status = db.Column(db.Boolean, default=False)  # 开关是否打开
    home_id = db.Column(db.Integer, db.ForeignKey('home.id'))   # 全屋的问题
    room_id = db.Column(db.Integer, db.ForeignKey('room.id'))  # 绑定的房间ID

    @classmethod
    def add_device(cls, d_mac, d_name, home_id, room_id):
        device_id = device_id_generator.generator_id()
        new_device = cls(id=device_id, name=d_name, mac=d_mac, home_id=home_id, room_id=room_id)
        try:
            db.session.add(new_device)
            db.session.commit()
            return new_device
        except SQLAlchemyError:
            db.session.rollback()
            return None

    def bind_appliance(self, e_type, e_brand, e_model, slot=0):
        return ElectricAppliance.bind_appliance(self.id, e_type, e_brand, e_model, slot)


class GroupDev(db.Model):
    __tablename__ = 'group_dev'

    id = db.Column(db.Integer, primary_key=True)
    nickname = db.Column(db.String, nullable=False)
    user_id = db.Column(db.Integer, db.ForeignKey('user.id'), nullable=False)

    @classmethod
    def add_group_dev(cls, nickname, user_id):
        _id = group_dev_id_generator.generator_id()
        new_group_dev = cls(id=_id, nickname=nickname, user_id=user_id)
        try:
            db.session.add(new_group_dev)
            db.session.commit()
            return new_group_dev
        except SQLAlchemyError:
            db.session.rollback()
            return None


device_groups = db.Table("device_group",
                         db.Column("device_id", db.Integer, db.ForeignKey('home.id')),
                         db.Column("groupdev_id", db.Integer, db.ForeignKey('group_dev.id'))
                         )


class ElectricAppliance(db.Model):
    __tablename__ = 'electric_appliance'

    id = db.Column(db.Integer, primary_key=True)
    device_id = db.Column(db.Integer, db.ForeignKey('device.id'))
    slot = db.Column(db.Integer)
    electric_type = db.Column(db.Integer)
    electric_brand = db.Column(db.String)
    electric_model = db.Column(db.String)

    @classmethod
    def bind_appliance(cls, device_id, e_type, e_brand, e_model, slot=0):
        el_id = appliance_id_generator.generator_id()
        electric_type = getattr(cls, e_type, 0)
        new_el = cls(
            id=el_id, device_id=device_id, slot=slot, electric_type=electric_type,
            electric_brand=e_brand, electric_model=e_model
        )
        try:
            db.session.add(new_el)
            db.session.commit()
            return new_el
        except SQLAlchemyError:
            db.session.rollback()
            return None


class SceneMode(db.Model):
    __tablename__ = 'scene_mode'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String)
    home_id = db.Column(db.Integer, db.ForeignKey('home.id'))

    @classmethod
    def add_scene(cls, name, home_id):
        new_scene = cls(id=scene_id_generator.generator_id(), name=name, home_id=home_id)
        try:
            db.session.add(new_scene)
            db.session.commit()
            return new_scene
        except SQLAlchemyError:
            db.session.rollback()
            return None

    def add_devices(self, devices: list):
        """
        场景中添加device
        :param devices: list [{"DeviceID": "", "Command": ""}]
        :return: bool
        """
        insert_list = []
        for device_info in devices:
            if (device_info.get("DeviceID") and str(device_info.get("DeviceID")).isnumeric()) \
                    and (device_info.get("Command") and str(device_info.get("Command")).upper() in ["ON", "OFF"]):
                insert_list.append({
                    "scene_id": self.id,
                    "device_id": device_info.get("DeviceID"),
                    "device_status": device_info.get("Command")
                })
        try:
            db.session.execute(scene_devices.insert().values(insert_list))
            db.session.commit()
            return True
        except SQLAlchemyError:
            db.session.rollback()
            return False


scene_devices = db.Table("scene_device",
                         db.Column("scene_id", db.Integer, db.ForeignKey("scene_mode.id")),
                         db.Column("device_id", db.Integer, db.ForeignKey("device.id")),
                         db.Column("device_status", db.String)
                         )


class Message(db.Model):
    __tablename__="message"

    SYSTEM_MSG = 1
    DEVICE_MSG = 2

    id = db.Column(db.Integer, primary_key=True)
    receiver_id = db.Column(db.Integer, db.ForeignKey("user.id"))
    title = db.Column(db.String)
    context = db.Column(db.Text)
    ext_data = db.Column(db.String)
    message_type = db.Column(db.Integer)
    is_read = db.Column(db.Boolean, default=False)
    create_time = db.Column(db.DateTime, default=datetime.datetime.now)

    @classmethod
    def add_message(cls, **kwargs):
        i = {
            "title": kwargs.get("title"),
            "context": kwargs.get("context"),
            "receiver_id": kwargs.get("receiver_id"),
            "message_type": kwargs.get("message_type"),
            "ext_data": kwargs.get("ext_data")
        }
        new_msg = cls(id=message_id_generator.generator_id(), **i)
        try:
            db.session.add(new_msg)
            db.session.commit()
            return new_msg
        except SQLAlchemyError:
            db.session.rollback()
            return None


class HomeSharing(db.Model):
    __tablename__ = "home_sharing"

    id = db.Column(db.Integer, primary_key=True)
    home_id = db.Column(db.Integer, db.ForeignKey('home.id'))
    share_from = db.Column(db.Integer, db.ForeignKey("user.id"))
    share_to = db.Column(db.Integer, db.ForeignKey("user.id"))
    permission = db.Column(db.Integer, default=Home.HOME_USER)
    create_time = db.Column(db.DateTime, default=datetime.datetime.now)

    @classmethod
    def create_share(cls, home_id, share_from, share_to, permission=Home.HOME_USER):
        new_share = cls(
            id=home_share_id_generator.generator_id(), home_id=home_id, share_from=share_from,
            share_to=share_to, permission=permission)
        try:
            db.session.add(new_share)
            db.session.commit()
            return new_share
        except SQLAlchemyError:
            db.session.rollback()
            return None


class DeviceSharing(db.Model):
    __tablename__ = "device_sharing"

    id = db.Column(db.Integer, primary_key=True)
    device_id = db.Column(db.Integer, db.ForeignKey('device.id'))
    share_from = db.Column(db.Integer, db.ForeignKey('user.id'))
    share_to = db.Column(db.Integer, db.ForeignKey('user.id'))
    create_time = db.Column(db.DateTime, default=datetime.datetime.now)

    @classmethod
    def create_share(cls, device_id, share_from, share_to):
        new_share = cls(
            id=device_share_id_generator.generator_id(), device_id=device_id, share_from=share_from, share_to=share_to)
        try:
            db.session.add(new_share)
            db.session.commit()
            return new_share
        except SQLAlchemyError:
            db.session.rollback()
            return None


shared_devices = db.Table("shared_device",
                          db.Column("device_id", db.Integer, db.ForeignKey("device.id")),
                          db.Column("share_from", db.Integer, db.ForeignKey("user.id")),
                          db.Column("share_to", db.Integer, db.ForeignKey("user.id")),
                          )
